"use client";
import { useState } from "react";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useAuth } from "@/context/AuthUserContext";
import ErrorAlert from "@/components/shared/alert/error-alert";
import SuccessAlert from "@/components/shared/alert/success-alert";
import GoogleSignIn from "@/components/authentication/google-signin";
import { Separator } from "@/components/ui/separator";
import { Loader2 } from "lucide-react";

export default function SignUp() {
  const [error, setError] = useState(null);
  const [success, setSuccess] = useState(null);
  const { createUserWithEmailAndPasswordHandler, loading, authUser } =
    useAuth();

  async function handleSubmit(event) {
    event.preventDefault();
    setError(null);
    const email = event.target.signupEmail.value;
    const password1 = event.target.password1.value;
    const password2 = event.target.password2.value;
    const name = event.target.signupName.value;

    // Validate the email
    if (!email) {
      setError("Email is required");
      return;
    }

    // Validate the password
    if (password1.length < 6) {
      setError("Password should be at least 6 characters");
      return;
    }

    if (password1 !== password2) {
      setError("Passwords do not match");
      return;
    }

    // Create the user
    try {
      const user = await createUserWithEmailAndPasswordHandler(
        email,
        password1,
        name
      );

      if (user) {
        // Verification email sent
        setSuccess("Verification email sent. Please check your inbox");
      }
    } catch (error) {
      setError(error.message);
    }
  }

  return (
    <Card className="mx-auto max-w-sm h-fit mb-8">
      <CardHeader>
        <CardTitle className="text-xl">Create Your Account</CardTitle>
        <CardDescription>
          Enter your information to create an account
        </CardDescription>
      </CardHeader>
      <CardContent>
        <div className="grid gap-6">
          {/* SignIn with Google Button */}
          <GoogleSignIn />
          <Separator />
          {/* Email and Password Form */}
          <form onSubmit={handleSubmit} className="grid gap-6">
            {error && <ErrorAlert message={error} />}
            {success && <SuccessAlert message={success} />}
            <div className="grid gap-2">
              <Label htmlFor="signupName">Name</Label>
              <Input
                id="signupName"
                type="text"
                placeholder="John Doe"
                required
              />
            </div>
            <div className="grid gap-2">
              <Label htmlFor="signupEmail">Email</Label>
              <Input
                id="signupEmail"
                type="signupEmail"
                placeholder="m@example.com"
                required
              />
            </div>
            <div className="grid gap-2">
              <Label htmlFor="password1">Password</Label>
              <Input id="password1" type="password" required />
            </div>
            <div className="grid gap-2">
              <Label htmlFor="password2">Confirm Password</Label>
              <Input id="password2" type="password" required />
            </div>
            <Button
              type="submit"
              className="w-full"
              disabled={loading || authUser}
            >
              {loading && <Loader2 className="w-6 h-6 mr-2 animate-spin" />}
              <span>Create an account</span>
            </Button>
          </form>
        </div>
        <div className="mt-4 text-center text-sm">
          Already have an account? Consider login
        </div>
      </CardContent>
    </Card>
  );
}
